-- ********************************************************************
-- configDB_SQL.SQL 
--
-- Configure CDRData database schema
--                                                                    
-- ********************************************************************

USE CDRData
GO

-- Add Settings if not there
IF NOT EXISTS (SELECT * FROM sysobjects WHERE name = N'Settings')
BEGIN
	create table Settings
	(
		Name                       varchar(255)          not null,
		Type                       smallint                      ,
		strValue                   varchar(255)                  ,
		dwValue                    integer                       ,
	)
END
GO

-- Add Workstations if not there
IF NOT EXISTS (SELECT * FROM sysobjects WHERE name = N'Workstations')
BEGIN
	create table Workstations
	(
		AETitle                    varchar(16)           not null,
		IPAddress                  varchar(255)                  ,
		Port                       varchar(255)                  ,
		LocalAE                    varchar(16)                   ,
		MachineName                varchar(255)                  ,
	)
END
GO

-- Add Volume if not there
IF NOT EXISTS (SELECT * FROM sysobjects WHERE name = N'Volume')
BEGIN
	create table Volume
	(
		VolumeNumber               integer       IDENTITY (1,1)        not null,
		UNCPathName                varchar(255)                  ,
		CDRom                      BIT                   not null
			default 0,
		Shared                     BIT                   not null
			default 0,
		VolumeName                 varchar(255)                  ,
		VolumeSerial               integer                       ,
		FileSetID                  varchar(16)                   ,
		FileSetUID                 varchar(64)                   ,
	)
END
GO

-- Add Patient if not there
IF NOT EXISTS (SELECT * FROM sysobjects WHERE name = N'Patient')
BEGIN
	create table Patient
	(
		PatientNumber              integer        IDENTITY (1,1)       not null,
		NameDICOM                  varchar(320)          not null,
		NameLast                   varchar(64)           not null,
		NameFirst                  varchar(64)                   ,
		PatientID                  varchar(64)                   ,
		PatientSex                 varchar(16)                   ,
		BirthDate                  DATETIME                      ,
		OtherID                    varchar(64)                   ,
		Comments                   text                  ,
		DeleteFlag                 BIT                   not null
			default 0,
		EnteredDate                DATETIME              not null
			default CURRENT_TIMESTAMP,
		ModifiedDate               DATETIME              not null
			default CURRENT_TIMESTAMP,
		PatientUID                 varchar(64)                   ,
	)
END
GO

IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'PatientNameID_IK'
       AND id = object_id(N'Patient'))
BEGIN
	create index PatientNameID_IK on Patient (NameLast, NameFirst , PatientID )
END
GO

-- Add Study if not there
IF NOT EXISTS (SELECT * FROM sysobjects WHERE name = N'Study')
BEGIN
	create table Study
	(
		StudyNumber                integer          IDENTITY (1,1)     not null,
		PatientNumber              integer                       ,
		StudyUID                   varchar(64)                   ,
		StudyDate                  DATETIME                      ,
		StudyID                    varchar(16)                   ,
		AccessionNumber            varchar(16)                   ,
		Description                varchar(64)                   ,
		Comments                   text                          ,
		ReferringPhysician         varchar(64)                   ,
		ViewSetUID                 varchar(64)                   ,
		ViewSetNumber              integer                       ,
		Exported                   BIT                   not null
			default 0,
		Imported                   BIT                   not null
			default 0,
		DeleteFlag                 BIT                   not null
			default 0,
		EnteredDate                DATETIME              not null
			default CURRENT_TIMESTAMP,
		ModifiedDate               DATETIME              not null
			default CURRENT_TIMESTAMP,
	)
END
GO

IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'StudyUID_IK'
       AND id = object_id(N'Study'))
BEGIN
	create unique index StudyUID_IK on Study (StudyUID asc)
END
GO

-- Add Series if not there
IF NOT EXISTS (SELECT * FROM sysobjects WHERE name = N'Series')
BEGIN
	create table Series
	(
		SeriesNumber               integer         IDENTITY (1,1)      not null,
		StudyNumber                integer                       ,
		Modality                   varchar(16)                   ,
		SeriesUID                  varchar(64)                   ,
		SeriesNumberDICOM          varchar(12)                   ,
		SeriesDate                 DATETIME                      ,
		PhysicianName              varchar(64)                   ,
		OperatorName               varchar(64)                   ,
		SeriesDescription          varchar(64)                   ,
		BodyPart                   varchar(16)                   ,
		Exported                   BIT                   not null
			default 0,
		Imported                   BIT                   not null
			default 0,
		DeleteFlag                 BIT                   not null
			default 0,
		EnteredDate                DATETIME              not null
			default CURRENT_TIMESTAMP,
		ModifiedDate               DATETIME              not null
			default CURRENT_TIMESTAMP,
	)
END
GO

IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'SeriesUID_IK'
       AND id = object_id(N'Series'))
BEGIN
	create unique index SeriesUID_IK on Series (SeriesUID asc)
END
GO

-- Add Image if not there
IF NOT EXISTS (SELECT * FROM sysobjects WHERE name = N'Image')
BEGIN
	create table Image
	(
		ImageNumber                integer          IDENTITY (1,1)     not null,
		SeriesNumber               integer                       ,
		ClassUID                   varchar(64)                   ,
		ImageUID                   varchar(64)                   ,
		ImageNumberDICOM           varchar(12)                   ,
		ImageType                  varchar(255)                  ,
		ImageFileName              varchar(255)                  ,
		ImageFileVolumeNumber      integer                       ,
		ImageSource                integer                       ,
		Width                      integer                       ,
		Height                     integer                       ,
		BitsPerPixel               integer                       ,
		SensorType                 varchar(255)                  ,
		SensorNumber               varchar(16)                   ,
		SensorMode                 varchar(255)                  ,
		MicronsPerPixel            integer                       ,
		MicronsPerPixelCalibrated  integer                       ,
		ImageDate                  DATETIME                      ,
		AcquiredDate               DATETIME                      ,
		CalibratedDate             DATETIME                      ,
		Rotation                   smallint                      ,
		HorizontalFlip             BIT                   not null
			default 0,
		Laterality                 varchar(16)                   ,
		AnatomicRegion             varchar(255)                  ,
		AnatomicRegionModifier     varchar(255)                  ,
		AnatomicStructure          varchar(255)                  ,
		AnatomicStructureModifier  varchar(255)                  ,
		Comments                   text                  ,
		Exported                   BIT                   not null
			default 0,
		Imported                   BIT                   not null
			default 0,
		DeleteFlag                 BIT                   not null
			default 0,
		EnteredDate                DATETIME              not null
			default CURRENT_TIMESTAMP,
		ModifiedDate               DATETIME              not null
			default CURRENT_TIMESTAMP,
		TransferSyntax             varchar(64)                   ,
		FileSize                   integer                       ,
	)
END
GO

-- Create primary keys and indexes 
IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'ImageUID_IK'
       AND id = object_id(N'Image'))
BEGIN
	create unique index ImageUID_IK on Image (ImageUID asc)
END
GO

IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'PK_Image'
       AND id = object_id(N'Image'))
BEGIN
	ALTER TABLE [dbo].[Image] WITH NOCHECK ADD 
		CONSTRAINT [PK_Image] PRIMARY KEY  NONCLUSTERED 
		(
			[ImageNumber]
		) ON [PRIMARY]
END
GO

IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'IX_Image'
       AND id = object_id(N'Image'))
BEGIN
	ALTER TABLE [dbo].[Image] WITH NOCHECK ADD 
		CONSTRAINT [IX_Image] UNIQUE  NONCLUSTERED 
		(
			[ImageUID]
		)  ON [PRIMARY] 
END
GO

IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'PK_Patient'
       AND id = object_id(N'Patient'))
BEGIN
	ALTER TABLE [dbo].[Patient] WITH NOCHECK ADD 
		CONSTRAINT [PK_Patient] PRIMARY KEY  NONCLUSTERED 
		(
			[PatientNumber]
		)  ON [PRIMARY] 
END
GO

IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'PK_Series'
       AND id = object_id(N'Series'))
BEGIN
	ALTER TABLE [dbo].[Series] WITH NOCHECK ADD 
		CONSTRAINT [PK_Series] PRIMARY KEY  NONCLUSTERED 
		(
			[SeriesNumber]
		)  ON [PRIMARY]
END
GO

IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'IX_Series'
       AND id = object_id(N'Series'))
BEGIN
	ALTER TABLE [dbo].[Series] WITH NOCHECK ADD 
		CONSTRAINT [IX_Series] UNIQUE  NONCLUSTERED 
		(
			[SeriesUID]
		)  ON [PRIMARY]
END
GO

IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'PK_Study'
       AND id = object_id(N'Study'))
BEGIN
	ALTER TABLE [dbo].[Study] WITH NOCHECK ADD 
		CONSTRAINT [PK_Study] PRIMARY KEY  NONCLUSTERED 
		(
			[StudyNumber]
		)  ON [PRIMARY]
END
GO

IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'IX_Study'
       AND id = object_id(N'Study'))
BEGIN
	ALTER TABLE [dbo].[Study] WITH NOCHECK ADD 
		CONSTRAINT [IX_Study] UNIQUE  NONCLUSTERED 
		(
			[StudyUID]
		)  ON [PRIMARY]
END
GO

IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'PK_Volume'
       AND id = object_id(N'Volume'))
BEGIN
	ALTER TABLE [dbo].[Volume] WITH NOCHECK ADD 
		CONSTRAINT [PK_Volume] PRIMARY KEY  NONCLUSTERED 
		(
			[VolumeNumber]
		)  ON [PRIMARY]
END
GO

IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'PK_Workstations'
       AND id = object_id(N'Workstations'))
BEGIN
	ALTER TABLE [dbo].[Workstations] WITH NOCHECK ADD 
		CONSTRAINT [PK_Workstations] PRIMARY KEY  NONCLUSTERED 
		(
			[AETitle]
		)  ON [PRIMARY] 
END
GO

IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'PK_Settings'
       AND id = object_id(N'Settings'))
BEGIN
	ALTER TABLE [dbo].[Settings] WITH NOCHECK ADD 
		CONSTRAINT [PK_Settings] PRIMARY KEY  NONCLUSTERED 
		(
			[Name]
		)  ON [PRIMARY] 
END
GO


-- Create foreign keys
IF NOT EXISTS (SELECT * FROM sys.foreign_keys WHERE object_id = OBJECT_ID(N'FK_Image_Series') 
	AND parent_object_id = OBJECT_ID(N'Image'))
BEGIN
	ALTER TABLE [dbo].[Image] ADD 
		CONSTRAINT [FK_Image_Series] FOREIGN KEY 
		(
			[SeriesNumber]
		) REFERENCES [dbo].[Series] (
			[SeriesNumber]
		)
END
GO

IF NOT EXISTS (SELECT * FROM sys.foreign_keys WHERE object_id = OBJECT_ID(N'FK_Image_Volume') 
	AND parent_object_id = OBJECT_ID(N'Image'))
BEGIN
	ALTER TABLE [dbo].[Image] ADD 
		CONSTRAINT [FK_Image_Volume] FOREIGN KEY 
		(
			[ImageFileVolumeNumber]
		) REFERENCES [dbo].[Volume] (
			[VolumeNumber]
		)
END
GO

IF NOT EXISTS (SELECT * FROM sys.foreign_keys WHERE object_id = OBJECT_ID(N'FK_Series_Study') 
	AND parent_object_id = OBJECT_ID(N'Series'))
BEGIN
	ALTER TABLE [dbo].[Series] ADD 
		CONSTRAINT [FK_Series_Study] FOREIGN KEY 
		(
			[StudyNumber]
		) REFERENCES [dbo].[Study] (
			[StudyNumber]
		)
END
GO

IF NOT EXISTS (SELECT * FROM sys.foreign_keys WHERE object_id = OBJECT_ID(N'FK_Study_Patient') 
	AND parent_object_id = OBJECT_ID(N'Study'))
BEGIN
	ALTER TABLE [dbo].[Study] ADD 
		CONSTRAINT [FK_Study_Patient] FOREIGN KEY 
		(
			[PatientNumber]
		) REFERENCES [dbo].[Patient] (
			[PatientNumber]
		)
END
GO

/* Additional indexes added in Version 4.0 */
IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'PatientDelete_IK'
       AND id = object_id(N'Patient'))
BEGIN
	create index PatientDelete_IK on Patient (DeleteFlag)
END
GO

IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'StudyDelete_IK'
       AND id = object_id(N'Study'))
BEGIN
	create index StudyDelete_IK on Study (DeleteFlag)
END
GO

IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'SeriesDelete_IK'
       AND id = object_id(N'Series'))
BEGIN
	create index SeriesDelete_IK on Series (DeleteFlag)
END
GO

IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'ImageDelete_IK'
       AND id = object_id(N'Image'))
BEGIN
	create index ImageDelete_IK on Image (DeleteFlag)
END
GO

IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'ImageVolumeNumber_IK'
       AND id = object_id(N'Image'))
BEGIN
	create index ImageVolumeNumber_IK on Image (ImageFileVolumeNumber asc)
END
GO

IF NOT EXISTS (SELECT * FROM sysindexes WHERE name = N'PatientUID_IK'
       AND id = object_id(N'Patient'))
BEGIN
	create index PatientUID_IK on Patient (PatientUID)
END
GO


/****** Create users/logins and rights ******/
-- make sure we have a login for the Users group
IF NOT EXISTS (SELECT * FROM sys.server_principals
    WHERE name = N'BUILTIN\Users' AND default_database_name = N'master')
BEGIN
	CREATE LOGIN [BUILTIN\Users] FROM WINDOWS WITH DEFAULT_DATABASE=[master], DEFAULT_LANGUAGE=[us_english]
END
GO

-- create a SQL user called cdr with password cdr for simple access to our database
IF NOT EXISTS (SELECT * FROM sys.server_principals
    WHERE name = N'cdr' AND default_database_name = N'CDRData')
BEGIN
	CREATE LOGIN [cdr] WITH PASSWORD=N'cdr', DEFAULT_DATABASE=[CDRData], DEFAULT_LANGUAGE=[us_english], CHECK_EXPIRATION=OFF, CHECK_POLICY=OFF
END
GO

-- now make sure the Users group and SQL cdr user have access to our database
IF NOT EXISTS (select * from sys.database_principals  where name = N'BUILTIN\Users')
BEGIN
	CREATE USER [BUILTIN\Users] FOR LOGIN [BUILTIN\Users]
END
GO

IF NOT EXISTS (select * from sys.database_principals  where name = N'cdr')
BEGIN
	CREATE USER [cdr] FOR LOGIN [cdr] WITH DEFAULT_SCHEMA=[dbo]
END
GO

-- MANTIS #1151: Create Login is not supported under SQL 2000 (version 8.0)
if (left(cast(serverproperty('productversion') as varchar), 4) LIKE '8.%')
BEGIN
		EXEC sp_grantlogin 'BUILTIN\Users'
		EXEC sp_grantdbaccess 'BUILTIN\Users'
		EXEC sp_addlogin 'cdr', 'cdr', 'CDRData'
		EXEC sp_adduser 'cdr', 'cdr'
END
GO


-- grant necessary rights to all tables for [public] role
GRANT  ALTER , SELECT ,  INSERT ,  UPDATE , DELETE  ON [dbo].[Patient]  TO [public]
GO
GRANT  ALTER , SELECT ,  INSERT ,  UPDATE , DELETE  ON [dbo].[Study]  TO [public]
GO
GRANT  ALTER , SELECT ,  INSERT ,  UPDATE , DELETE  ON [dbo].[Series]  TO [public]
GO
GRANT  ALTER , SELECT ,  INSERT ,  UPDATE , DELETE  ON [dbo].[Image]  TO [public]
GO
GRANT  ALTER , SELECT ,  INSERT ,  UPDATE , DELETE  ON [dbo].[Volume]  TO [public]
GO
GRANT  ALTER , SELECT ,  INSERT ,  UPDATE , DELETE  ON [dbo].[Workstations]  TO [public]
GO
GRANT  ALTER , SELECT ,  INSERT ,  UPDATE , DELETE  ON [dbo].[Settings]  TO [public]
GO

/* Enter DB Schema Version Info */
IF NOT EXISTS (SELECT * FROM Settings WHERE Name = N'Schema')
BEGIN
	INSERT INTO [dbo].[Settings] (Name, Type, strValue, dwValue) 
		VALUES ('Schema', 2, '5.0.1305.4336', 50004336)
END
ELSE
BEGIN
	UPDATE [dbo].[Settings]
	   SET [Name] = 'Schema'
		  ,[Type] = 2
		  ,[strValue] = '5.0.1305.4336'
		  ,[dwValue] = 50004336
	 WHERE Name = 'Schema'
END
GO